import { getPool } from '../../db/pool.js';
export class FeeRepository {
    // ================================================
    // Fee Arrangements
    // ================================================
    async getFeeArrangementByMatter(matterId) {
        const pool = await getPool();
        const query = `
      SELECT 
        fa.*,
        m.matter_number,
        m.title as matter_title,
        creator.username as created_by_name
      FROM fee_arrangements fa
      JOIN matters m ON fa.matter_id = m.id
      JOIN users creator ON fa.created_by = creator.id
      WHERE fa.matter_id = ? AND fa.is_active = TRUE
      ORDER BY fa.effective_date DESC
      LIMIT 1
    `;
        const [rows] = await pool.query(query, [matterId]);
        return rows.length > 0 ? rows[0] : null;
    }
    async getFeeArrangementById(id) {
        const pool = await getPool();
        const query = `
      SELECT 
        fa.*,
        m.matter_number,
        m.title as matter_title,
        creator.username as created_by_name
      FROM fee_arrangements fa
      JOIN matters m ON fa.matter_id = m.id
      JOIN users creator ON fa.created_by = creator.id
      WHERE fa.id = ?
    `;
        const [rows] = await pool.query(query, [id]);
        return rows.length > 0 ? rows[0] : null;
    }
    async listFeeArrangements(matterId) {
        const pool = await getPool();
        let whereClause = '';
        const params = [];
        if (matterId) {
            whereClause = 'WHERE fa.matter_id = ?';
            params.push(matterId);
        }
        const query = `
      SELECT 
        fa.*,
        m.matter_number,
        m.title as matter_title,
        creator.username as created_by_name
      FROM fee_arrangements fa
      JOIN matters m ON fa.matter_id = m.id
      JOIN users creator ON fa.created_by = creator.id
      ${whereClause}
      ORDER BY fa.created_at DESC
    `;
        const [rows] = await pool.query(query, params);
        return rows;
    }
    async createFeeArrangement(data, userId) {
        const pool = await getPool();
        // Check if matter exists
        const [matters] = await pool.query('SELECT id FROM matters WHERE id = ?', [data.matter_id]);
        if (matters.length === 0) {
            throw new Error('Matter not found');
        }
        // Deactivate existing arrangements for this matter
        await pool.query('UPDATE fee_arrangements SET is_active = FALSE WHERE matter_id = ?', [data.matter_id]);
        const query = `
      INSERT INTO fee_arrangements (
        matter_id, fee_type, base_hourly_rate, fixed_amount, cap_amount,
        contingency_percent, billing_frequency, payment_terms_days,
        description, terms_document_path, effective_date, end_date,
        created_by
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `;
        const [result] = await pool.query(query, [
            data.matter_id,
            data.fee_type,
            data.base_hourly_rate || null,
            data.fixed_amount || null,
            data.cap_amount || null,
            data.contingency_percent || null,
            data.billing_frequency,
            data.payment_terms_days,
            data.description || null,
            data.terms_document_path || null,
            data.effective_date,
            data.end_date || null,
            userId,
        ]);
        return this.getFeeArrangementById(result.insertId);
    }
    async updateFeeArrangement(id, data) {
        const pool = await getPool();
        // Check if arrangement exists
        const existing = await this.getFeeArrangementById(id);
        if (!existing) {
            throw new Error('Fee arrangement not found');
        }
        const updates = [];
        const params = [];
        if (data.fee_type !== undefined) {
            updates.push('fee_type = ?');
            params.push(data.fee_type);
        }
        if (data.base_hourly_rate !== undefined) {
            updates.push('base_hourly_rate = ?');
            params.push(data.base_hourly_rate);
        }
        if (data.fixed_amount !== undefined) {
            updates.push('fixed_amount = ?');
            params.push(data.fixed_amount);
        }
        if (data.cap_amount !== undefined) {
            updates.push('cap_amount = ?');
            params.push(data.cap_amount);
        }
        if (data.contingency_percent !== undefined) {
            updates.push('contingency_percent = ?');
            params.push(data.contingency_percent);
        }
        if (data.billing_frequency !== undefined) {
            updates.push('billing_frequency = ?');
            params.push(data.billing_frequency);
        }
        if (data.payment_terms_days !== undefined) {
            updates.push('payment_terms_days = ?');
            params.push(data.payment_terms_days);
        }
        if (data.description !== undefined) {
            updates.push('description = ?');
            params.push(data.description);
        }
        if (data.terms_document_path !== undefined) {
            updates.push('terms_document_path = ?');
            params.push(data.terms_document_path);
        }
        if (data.end_date !== undefined) {
            updates.push('end_date = ?');
            params.push(data.end_date);
        }
        if (data.is_active !== undefined) {
            updates.push('is_active = ?');
            params.push(data.is_active);
        }
        if (updates.length === 0) {
            return existing;
        }
        const query = `UPDATE fee_arrangements SET ${updates.join(', ')} WHERE id = ?`;
        await pool.query(query, [...params, id]);
        return this.getFeeArrangementById(id);
    }
    async deleteFeeArrangement(id) {
        const pool = await getPool();
        const existing = await this.getFeeArrangementById(id);
        if (!existing) {
            throw new Error('Fee arrangement not found');
        }
        // Soft delete by setting is_active to false
        await pool.query('UPDATE fee_arrangements SET is_active = FALSE WHERE id = ?', [id]);
    }
    // ================================================
    // Rate Cards
    // ================================================
    async listRateCards(query) {
        const pool = await getPool();
        const { user_id, role_id, practice_area, client_id, is_default } = query;
        let whereConditions = [];
        let params = [];
        if (user_id) {
            whereConditions.push('rc.user_id = ?');
            params.push(user_id);
        }
        if (role_id) {
            whereConditions.push('rc.role_id = ?');
            params.push(role_id);
        }
        if (practice_area) {
            whereConditions.push('rc.practice_area = ?');
            params.push(practice_area);
        }
        if (client_id) {
            whereConditions.push('rc.client_id = ?');
            params.push(client_id);
        }
        if (is_default !== undefined) {
            whereConditions.push('rc.is_default = ?');
            params.push(is_default);
        }
        // Only show active rate cards (end_date is null or in the future)
        whereConditions.push('(rc.end_date IS NULL OR rc.end_date >= CURDATE())');
        const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';
        const sql = `
      SELECT 
        rc.*,
        u.username as user_name,
        r.role_name,
        c.name as client_name,
        creator.username as created_by_name
      FROM rate_cards rc
      LEFT JOIN users u ON rc.user_id = u.id
      LEFT JOIN roles r ON rc.role_id = r.id
      LEFT JOIN clients c ON rc.client_id = c.id
      JOIN users creator ON rc.created_by = creator.id
      ${whereClause}
      ORDER BY rc.effective_date DESC, rc.created_at DESC
    `;
        const [rows] = await pool.query(sql, params);
        return rows;
    }
    async getRateCardById(id) {
        const pool = await getPool();
        const query = `
      SELECT 
        rc.*,
        u.username as user_name,
        r.role_name,
        c.name as client_name,
        creator.username as created_by_name
      FROM rate_cards rc
      LEFT JOIN users u ON rc.user_id = u.id
      LEFT JOIN roles r ON rc.role_id = r.id
      LEFT JOIN clients c ON rc.client_id = c.id
      JOIN users creator ON rc.created_by = creator.id
      WHERE rc.id = ?
    `;
        const [rows] = await pool.query(query, [id]);
        return rows.length > 0 ? rows[0] : null;
    }
    async createRateCard(data, userId) {
        const pool = await getPool();
        // Validate at least one of user_id, role_id, practice_area, or client_id is provided
        if (!data.user_id && !data.role_id && !data.practice_area && !data.client_id) {
            throw new Error('At least one of user_id, role_id, practice_area, or client_id must be provided');
        }
        const query = `
      INSERT INTO rate_cards (
        user_id, role_id, practice_area, client_id,
        hourly_rate, currency, effective_date, end_date,
        is_default, notes, created_by
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `;
        const [result] = await pool.query(query, [
            data.user_id || null,
            data.role_id || null,
            data.practice_area || null,
            data.client_id || null,
            data.hourly_rate,
            data.currency,
            data.effective_date,
            data.end_date || null,
            data.is_default,
            data.notes || null,
            userId,
        ]);
        return this.getRateCardById(result.insertId);
    }
    async updateRateCard(id, data) {
        const pool = await getPool();
        const existing = await this.getRateCardById(id);
        if (!existing) {
            throw new Error('Rate card not found');
        }
        const updates = [];
        const params = [];
        if (data.hourly_rate !== undefined) {
            updates.push('hourly_rate = ?');
            params.push(data.hourly_rate);
        }
        if (data.end_date !== undefined) {
            updates.push('end_date = ?');
            params.push(data.end_date);
        }
        if (data.is_default !== undefined) {
            updates.push('is_default = ?');
            params.push(data.is_default);
        }
        if (data.notes !== undefined) {
            updates.push('notes = ?');
            params.push(data.notes);
        }
        if (updates.length === 0) {
            return existing;
        }
        const query = `UPDATE rate_cards SET ${updates.join(', ')} WHERE id = ?`;
        await pool.query(query, [...params, id]);
        return this.getRateCardById(id);
    }
    async deleteRateCard(id) {
        const pool = await getPool();
        const existing = await this.getRateCardById(id);
        if (!existing) {
            throw new Error('Rate card not found');
        }
        await pool.query('DELETE FROM rate_cards WHERE id = ?', [id]);
    }
    // ================================================
    // Helper Methods
    // ================================================
    async getApplicableRate(userId, matterId) {
        const pool = await getPool();
        // Priority order:
        // 1. User + Client rate
        // 2. User + Matter rate
        // 3. User default rate
        // 4. Role + Client rate
        // 5. Role default rate
        // 6. Practice area rate
        // 7. Default rate
        const query = `
      SELECT rc.hourly_rate
      FROM rate_cards rc
      LEFT JOIN matters m ON m.id = ?
      WHERE rc.user_id = ? 
        AND (rc.client_id = m.client_id OR rc.client_id IS NULL)
        AND (rc.end_date IS NULL OR rc.end_date >= CURDATE())
        AND rc.effective_date <= CURDATE()
      ORDER BY 
        CASE WHEN rc.client_id IS NOT NULL THEN 1 ELSE 2 END,
        rc.effective_date DESC
      LIMIT 1
    `;
        const [rows] = await pool.query(query, [matterId, userId]);
        if (rows.length > 0) {
            return rows[0].hourly_rate;
        }
        return null;
    }
}
export const feeRepo = new FeeRepository();
//# sourceMappingURL=repo.js.map