import { z } from 'zod';
// ================================================
// Invoice DTOs
// ================================================
// Create Invoice Line Item
export const createInvoiceLineItemSchema = z.object({
    item_type: z.enum(['Time', 'Expense', 'Fee', 'Adjustment']),
    time_entry_id: z.number().int().positive().optional(),
    expense_id: z.number().int().positive().optional(),
    description: z.string().min(1).max(5000),
    quantity: z.number().positive().default(1),
    unit_price: z.number(),
    amount: z.number(),
    is_taxable: z.boolean().default(true),
});
// Create Invoice
export const createInvoiceSchema = z.object({
    matter_id: z.number().int().positive(),
    client_id: z.number().int().positive(),
    invoice_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
    due_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
    time_entry_ids: z.array(z.number().int().positive()).optional(),
    expense_ids: z.array(z.number().int().positive()).optional(),
    line_items: z.array(createInvoiceLineItemSchema).optional(),
    notes: z.string().max(5000).optional(),
    payment_instructions: z.string().max(2000).optional(),
    apply_vat: z.boolean().default(true),
    apply_nhil: z.boolean().default(true),
    apply_getfund: z.boolean().default(true),
    apply_wht: z.boolean().default(false),
    wht_rate: z.number().min(0).max(100).default(7.5),
});
// Update Invoice
export const updateInvoiceSchema = z.object({
    invoice_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    due_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    notes: z.string().max(5000).optional(),
    payment_instructions: z.string().max(2000).optional(),
    status: z.enum(['Draft', 'Sent', 'Paid', 'Partially Paid', 'Overdue', 'Cancelled']).optional(),
});
// Query Parameters
export const queryInvoicesSchema = z.object({
    page: z.string().optional().transform(val => val ? parseInt(val) : 1),
    limit: z.string().optional().transform(val => val ? parseInt(val) : 50),
    matter_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
    client_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
    status: z.enum(['Draft', 'Sent', 'Paid', 'Partially Paid', 'Overdue', 'Cancelled']).optional(),
    start_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    overdue_only: z.string().optional().transform(val => val === 'true'),
});
// Send Invoice
export const sendInvoiceSchema = z.object({
    send_email: z.boolean().default(true),
    email_message: z.string().max(2000).optional(),
});
// Record Payment
export const recordPaymentSchema = z.object({
    amount: z.number().positive(),
    payment_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
    payment_method: z.string().max(50).optional(),
    reference_number: z.string().max(100).optional(),
    notes: z.string().max(1000).optional(),
});
//# sourceMappingURL=dto.js.map