import { MatterRepository } from './repo.js';
import { getPool } from '../../db/pool.js';
export class MatterService {
    repo;
    constructor() {
        this.repo = new MatterRepository();
    }
    /**
     * Generate matter ID based on configured pattern
     * Patterns supported:
     * - YEAR-DEPT-SEQ: 2025-LIT-0001
     * - DEPT-YEAR-SEQ: LIT-2025-0001
     * - SEQ-YEAR-DEPT: 0001-2025-LIT
     * - Custom with prefix
     */
    async generateMatterId(departmentId) {
        const { current, config } = await this.repo.getNextSequence();
        const year = new Date().getFullYear().toString();
        const sequence = current.toString().padStart(config.sequence_padding, '0');
        let deptCode = '';
        if (config.include_dept && departmentId) {
            // Get department code
            const pool = await getPool();
            const [dept] = await pool.query(`SELECT name FROM departments WHERE id = ? LIMIT 1`, [departmentId]);
            if (dept.length > 0) {
                // Use first 3 letters of department name as code
                deptCode = dept[0].name.substring(0, 3).toUpperCase();
            }
        }
        const parts = [];
        // Add prefix if configured
        if (config.prefix) {
            parts.push(config.prefix);
        }
        // Parse pattern and build matter ID
        const pattern = config.pattern.toUpperCase();
        if (pattern === 'YEAR-DEPT-SEQ') {
            if (config.include_year)
                parts.push(year);
            if (config.include_dept && deptCode)
                parts.push(deptCode);
            parts.push(sequence);
        }
        else if (pattern === 'DEPT-YEAR-SEQ') {
            if (config.include_dept && deptCode)
                parts.push(deptCode);
            if (config.include_year)
                parts.push(year);
            parts.push(sequence);
        }
        else if (pattern === 'SEQ-YEAR-DEPT') {
            parts.push(sequence);
            if (config.include_year)
                parts.push(year);
            if (config.include_dept && deptCode)
                parts.push(deptCode);
        }
        else {
            // Default: just sequence
            parts.push(sequence);
        }
        return parts.join(config.sep_char);
    }
    /**
     * Get department ID from lawyer
     */
    async getDepartmentIdFromLawyer(lawyerId) {
        const pool = await getPool();
        const [result] = await pool.query(`SELECT e.department_id 
       FROM lawyers l
       JOIN employees e ON e.id = l.employee_id
       WHERE l.id = ?
       LIMIT 1`, [lawyerId]);
        return result.length > 0 ? result[0].department_id : null;
    }
}
//# sourceMappingURL=service.js.map