import { z } from 'zod';
// ================================================
// Receipt DTOs
// ================================================
// Payment Allocation Item
export const paymentAllocationSchema = z.object({
    invoice_id: z.number().int().positive(),
    amount: z.number().positive(),
});
// Create Receipt
export const createReceiptSchema = z.object({
    client_id: z.number().int().positive(),
    receipt_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
    amount: z.number().positive(),
    payment_method: z.enum(['Cash', 'Cheque', 'Bank Transfer', 'Mobile Money', 'Card', 'Other']),
    reference_number: z.string().max(100).optional(),
    bank_name: z.string().max(100).optional(),
    cheque_number: z.string().max(50).optional(),
    notes: z.string().max(2000).optional(),
    allocations: z.array(paymentAllocationSchema).min(1),
});
// Update Receipt
export const updateReceiptSchema = z.object({
    receipt_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    payment_method: z.enum(['Cash', 'Cheque', 'Bank Transfer', 'Mobile Money', 'Card', 'Other']).optional(),
    reference_number: z.string().max(100).optional(),
    bank_name: z.string().max(100).optional(),
    cheque_number: z.string().max(50).optional(),
    notes: z.string().max(2000).optional(),
    status: z.enum(['Pending', 'Cleared', 'Bounced', 'Cancelled']).optional(),
});
// Query Parameters
export const queryReceiptsSchema = z.object({
    page: z.string().optional().transform(val => val ? parseInt(val) : 1),
    limit: z.string().optional().transform(val => val ? parseInt(val) : 50),
    client_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
    payment_method: z.enum(['Cash', 'Cheque', 'Bank Transfer', 'Mobile Money', 'Card', 'Other']).optional(),
    status: z.enum(['Pending', 'Cleared', 'Bounced', 'Cancelled']).optional(),
    start_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
});
// ================================================
// Collection Activity DTOs
// ================================================
// Create Collection Activity
export const createCollectionActivitySchema = z.object({
    client_id: z.number().int().positive(),
    invoice_id: z.number().int().positive().optional(),
    activity_type: z.enum(['Email', 'Phone', 'Letter', 'Meeting', 'Legal Action', 'Other']),
    activity_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
    contact_person: z.string().max(200).optional(),
    notes: z.string().max(5000),
    outcome: z.string().max(1000).optional(),
    follow_up_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
});
// Update Collection Activity
export const updateCollectionActivitySchema = z.object({
    activity_type: z.enum(['Email', 'Phone', 'Letter', 'Meeting', 'Legal Action', 'Other']).optional(),
    activity_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    contact_person: z.string().max(200).optional(),
    notes: z.string().max(5000).optional(),
    outcome: z.string().max(1000).optional(),
    follow_up_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
});
// Query Collection Activities
export const queryCollectionActivitiesSchema = z.object({
    page: z.string().optional().transform(val => val ? parseInt(val) : 1),
    limit: z.string().optional().transform(val => val ? parseInt(val) : 50),
    client_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
    invoice_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
    activity_type: z.enum(['Email', 'Phone', 'Letter', 'Meeting', 'Legal Action', 'Other']).optional(),
    start_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
});
//# sourceMappingURL=dto.js.map