import { getPool } from '../../db/pool.js';
export class RoleRepository {
    /**
     * Get all roles with department info
     */
    async findAll(departmentId) {
        const pool = await getPool();
        let query = `
      SELECT r.id, r.name, r.department_id, d.name as department_name, r.description, r.created_at
      FROM roles r
      JOIN departments d ON d.id = r.department_id
    `;
        const params = [];
        if (departmentId) {
            query += ' WHERE r.department_id = ?';
            params.push(departmentId);
        }
        query += ' ORDER BY d.name ASC, r.name ASC';
        const [rows] = await pool.query(query, params);
        return rows;
    }
    /**
     * Get role by ID
     */
    async findById(id) {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT r.id, r.name, r.department_id, d.name as department_name, r.description, r.created_at
       FROM roles r
       JOIN departments d ON d.id = r.department_id
       WHERE r.id = ?`, [id]);
        return rows.length > 0 ? rows[0] : null;
    }
    /**
     * Check if role name exists in department (excluding specific ID for updates)
     */
    async nameExistsInDepartment(name, departmentId, excludeId) {
        const pool = await getPool();
        const query = excludeId
            ? 'SELECT COUNT(*) as count FROM roles WHERE name = ? AND department_id = ? AND id != ?'
            : 'SELECT COUNT(*) as count FROM roles WHERE name = ? AND department_id = ?';
        const params = excludeId ? [name, departmentId, excludeId] : [name, departmentId];
        const [rows] = await pool.query(query, params);
        return rows[0].count > 0;
    }
    /**
     * Check if department exists
     */
    async departmentExists(departmentId) {
        const pool = await getPool();
        const [rows] = await pool.query('SELECT COUNT(*) as count FROM departments WHERE id = ?', [departmentId]);
        return rows[0].count > 0;
    }
    /**
     * Create role
     */
    async create(data) {
        const pool = await getPool();
        const [result] = await pool.query(`INSERT INTO roles (name, department_id, description)
       VALUES (?, ?, ?)`, [data.name, data.department_id, data.description || null]);
        const newRole = await this.findById(result.insertId);
        if (!newRole) {
            throw new Error('Failed to create role');
        }
        return newRole;
    }
    /**
     * Update role
     */
    async update(id, data) {
        const pool = await getPool();
        const updates = [];
        const values = [];
        if (data.name !== undefined) {
            updates.push('name = ?');
            values.push(data.name);
        }
        if (data.department_id !== undefined) {
            updates.push('department_id = ?');
            values.push(data.department_id);
        }
        if (data.description !== undefined) {
            updates.push('description = ?');
            values.push(data.description);
        }
        if (updates.length === 0) {
            const role = await this.findById(id);
            if (!role) {
                throw new Error('Role not found');
            }
            return role;
        }
        values.push(id);
        await pool.query(`UPDATE roles SET ${updates.join(', ')} WHERE id = ?`, values);
        const updated = await this.findById(id);
        if (!updated) {
            throw new Error('Role not found');
        }
        return updated;
    }
    /**
     * Delete role
     */
    async delete(id) {
        const pool = await getPool();
        await pool.query('DELETE FROM roles WHERE id = ?', [id]);
    }
    /**
     * Check if role has associated users
     */
    async hasUsers(id) {
        const pool = await getPool();
        const [rows] = await pool.query('SELECT COUNT(*) as count FROM users WHERE role_id = ?', [id]);
        return rows[0].count > 0;
    }
    /**
     * Log role action to audit log
     */
    async logAction(userId, roleId, action, meta) {
        const pool = await getPool();
        await pool.query(`INSERT INTO audit_logs (user_id, entity, entity_id, action, meta)
       VALUES (?, 'role', ?, ?, ?)`, [userId, roleId, action, JSON.stringify(meta)]);
    }
}
//# sourceMappingURL=repo.js.map