import { z } from 'zod';
// ================================================
// Time Entry DTOs
// ================================================
// Create Time Entry
export const createTimeEntrySchema = z.object({
    matter_id: z.number().int().positive(),
    task_id: z.number().int().positive().optional(),
    entry_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
    hours: z.number().positive().max(24),
    activity_code: z.string().min(1).max(50),
    description: z.string().min(1).max(5000),
    billable: z.boolean().default(true),
    hourly_rate: z.number().positive().optional(),
});
// Update Time Entry
export const updateTimeEntrySchema = z.object({
    entry_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    hours: z.number().positive().max(24).optional(),
    activity_code: z.string().min(1).max(50).optional(),
    description: z.string().min(1).max(5000).optional(),
    billable: z.boolean().optional(),
    hourly_rate: z.number().positive().optional(),
});
// Query Parameters
export const queryTimeEntriesSchema = z.object({
    page: z.string().optional().transform(val => val ? parseInt(val) : 1),
    limit: z.string().optional().transform(val => val ? parseInt(val) : 50),
    user_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
    matter_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
    status: z.enum(['Draft', 'Submitted', 'Approved', 'Rejected', 'Invoiced']).optional(),
    start_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    billable: z.string().optional().transform(val => val === 'true' ? true : val === 'false' ? false : undefined),
});
// Submit for Approval
export const submitTimeEntrySchema = z.object({
    entry_ids: z.array(z.number().int().positive()).min(1),
});
// Approve Time Entry
export const approveTimeEntrySchema = z.object({
    entry_ids: z.array(z.number().int().positive()).min(1),
});
// Reject Time Entry
export const rejectTimeEntrySchema = z.object({
    entry_ids: z.array(z.number().int().positive()).min(1),
    rejection_reason: z.string().min(1).max(1000),
});
// ================================================
// Timer DTOs
// ================================================
// Start Timer
export const startTimerSchema = z.object({
    matter_id: z.number().int().positive(),
    activity_code: z.string().min(1).max(50),
    description: z.string().max(5000).optional(),
});
// Stop Timer (no body needed, uses user context)
// ================================================
// Activity Code DTOs
// ================================================
// Create Activity Code (admin only)
export const createActivityCodeSchema = z.object({
    code: z.string().min(1).max(50),
    name: z.string().min(1).max(100),
    description: z.string().max(500).optional(),
    default_billable: z.boolean().default(true),
    sort_order: z.number().int().default(0),
});
// Update Activity Code
export const updateActivityCodeSchema = z.object({
    name: z.string().min(1).max(100).optional(),
    description: z.string().max(500).optional(),
    default_billable: z.boolean().optional(),
    is_active: z.boolean().optional(),
    sort_order: z.number().int().optional(),
});
//# sourceMappingURL=dto.js.map