import { getPool } from '../../db/pool.js';
export class TimeRepository {
    // ================================================
    // Time Entries
    // ================================================
    async listTimeEntries(query, currentUserId, canViewAll) {
        const pool = await getPool();
        const { page = 1, limit = 50, user_id, matter_id, status, start_date, end_date, billable } = query;
        const offset = (page - 1) * limit;
        let whereConditions = [];
        let params = [];
        // Permission check: users can only see their own entries unless they have view-all permission
        if (!canViewAll) {
            whereConditions.push('te.user_id = ?');
            params.push(currentUserId);
        }
        else if (user_id) {
            whereConditions.push('te.user_id = ?');
            params.push(user_id);
        }
        if (matter_id) {
            whereConditions.push('te.matter_id = ?');
            params.push(matter_id);
        }
        if (status) {
            whereConditions.push('te.status = ?');
            params.push(status);
        }
        if (start_date) {
            whereConditions.push('te.entry_date >= ?');
            params.push(start_date);
        }
        if (end_date) {
            whereConditions.push('te.entry_date <= ?');
            params.push(end_date);
        }
        if (billable !== undefined) {
            whereConditions.push('te.billable = ?');
            params.push(billable);
        }
        const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';
        // Get total count
        const countQuery = `
      SELECT COUNT(*) as total
      FROM time_entries te
      ${whereClause}
    `;
        const [countRows] = await pool.query(countQuery, params);
        const total = countRows[0].total;
        // Get entries
        const query_sql = `
      SELECT 
        te.*,
        u.username as user_name,
        m.matter_number,
        m.title as matter_title,
        ac.name as activity_name,
        approver.username as approved_by_name
      FROM time_entries te
      JOIN users u ON te.user_id = u.id
      JOIN matters m ON te.matter_id = m.id
      LEFT JOIN activity_codes ac ON te.activity_code = ac.code
      LEFT JOIN users approver ON te.approved_by = approver.id
      ${whereClause}
      ORDER BY te.entry_date DESC, te.created_at DESC
      LIMIT ? OFFSET ?
    `;
        const [rows] = await pool.query(query_sql, [...params, limit, offset]);
        return {
            entries: rows,
            pagination: {
                page,
                limit,
                total,
                totalPages: Math.ceil(total / limit),
            },
        };
    }
    async getTimeEntryById(id, currentUserId, canViewAll) {
        const pool = await getPool();
        const query = `
      SELECT 
        te.*,
        u.username as user_name,
        m.matter_number,
        m.title as matter_title,
        ac.name as activity_name,
        approver.username as approved_by_name
      FROM time_entries te
      JOIN users u ON te.user_id = u.id
      JOIN matters m ON te.matter_id = m.id
      LEFT JOIN activity_codes ac ON te.activity_code = ac.code
      LEFT JOIN users approver ON te.approved_by = approver.id
      WHERE te.id = ?
    `;
        const [rows] = await pool.query(query, [id]);
        if (rows.length === 0) {
            return null;
        }
        const entry = rows[0];
        // Permission check
        if (!canViewAll && entry.user_id !== currentUserId) {
            throw new Error('Forbidden: Cannot view other users time entries');
        }
        return entry;
    }
    async createTimeEntry(data, userId) {
        const pool = await getPool();
        // Calculate total amount if hourly_rate is provided
        let total_amount = null;
        if (data.hourly_rate) {
            total_amount = data.hours * data.hourly_rate;
        }
        const query = `
      INSERT INTO time_entries (
        user_id, matter_id, task_id, entry_date, hours, 
        activity_code, description, billable, hourly_rate, total_amount,
        created_by
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `;
        const [result] = await pool.query(query, [
            userId,
            data.matter_id,
            data.task_id || null,
            data.entry_date,
            data.hours,
            data.activity_code,
            data.description,
            data.billable,
            data.hourly_rate || null,
            total_amount,
            userId,
        ]);
        return this.getTimeEntryById(result.insertId, userId, true);
    }
    async updateTimeEntry(id, data, userId, canViewAll) {
        const pool = await getPool();
        // Check if entry exists and user has permission
        const existing = await this.getTimeEntryById(id, userId, canViewAll);
        if (!existing) {
            throw new Error('Time entry not found');
        }
        // Only allow updates if status is Draft or Rejected
        if (existing.status !== 'Draft' && existing.status !== 'Rejected') {
            throw new Error('Cannot update time entry that has been submitted or approved');
        }
        // Only allow user to update their own entries
        if (existing.user_id !== userId) {
            throw new Error('Forbidden: Cannot update other users time entries');
        }
        const updates = [];
        const params = [];
        if (data.entry_date !== undefined) {
            updates.push('entry_date = ?');
            params.push(data.entry_date);
        }
        if (data.hours !== undefined) {
            updates.push('hours = ?');
            params.push(data.hours);
        }
        if (data.activity_code !== undefined) {
            updates.push('activity_code = ?');
            params.push(data.activity_code);
        }
        if (data.description !== undefined) {
            updates.push('description = ?');
            params.push(data.description);
        }
        if (data.billable !== undefined) {
            updates.push('billable = ?');
            params.push(data.billable);
        }
        if (data.hourly_rate !== undefined) {
            updates.push('hourly_rate = ?');
            params.push(data.hourly_rate);
            // Recalculate total_amount
            const hours = data.hours !== undefined ? data.hours : existing.hours;
            updates.push('total_amount = ?');
            params.push(hours * data.hourly_rate);
        }
        if (updates.length === 0) {
            return existing;
        }
        const query = `UPDATE time_entries SET ${updates.join(', ')} WHERE id = ?`;
        await pool.query(query, [...params, id]);
        return this.getTimeEntryById(id, userId, true);
    }
    async deleteTimeEntry(id, userId, canViewAll) {
        const pool = await getPool();
        const existing = await this.getTimeEntryById(id, userId, canViewAll);
        if (!existing) {
            throw new Error('Time entry not found');
        }
        // Only allow deletion if status is Draft
        if (existing.status !== 'Draft') {
            throw new Error('Cannot delete time entry that has been submitted');
        }
        // Only allow user to delete their own entries
        if (existing.user_id !== userId) {
            throw new Error('Forbidden: Cannot delete other users time entries');
        }
        await pool.query('DELETE FROM time_entries WHERE id = ?', [id]);
    }
    async submitTimeEntries(entryIds, userId) {
        const pool = await getPool();
        // Verify all entries belong to user and are in Draft status
        const checkQuery = `
      SELECT id, status, user_id 
      FROM time_entries 
      WHERE id IN (${entryIds.map(() => '?').join(',')})
    `;
        const [entries] = await pool.query(checkQuery, entryIds);
        for (const entry of entries) {
            if (entry.user_id !== userId) {
                throw new Error(`Entry ${entry.id} does not belong to you`);
            }
            if (entry.status !== 'Draft') {
                throw new Error(`Entry ${entry.id} has already been submitted`);
            }
        }
        const updateQuery = `
      UPDATE time_entries 
      SET status = 'Submitted', submitted_at = NOW()
      WHERE id IN (${entryIds.map(() => '?').join(',')})
    `;
        await pool.query(updateQuery, entryIds);
    }
    async approveTimeEntries(entryIds, approverId) {
        const pool = await getPool();
        // Verify all entries are in Submitted status
        const checkQuery = `
      SELECT id, status 
      FROM time_entries 
      WHERE id IN (${entryIds.map(() => '?').join(',')})
    `;
        const [entries] = await pool.query(checkQuery, entryIds);
        for (const entry of entries) {
            if (entry.status !== 'Submitted') {
                throw new Error(`Entry ${entry.id} is not in Submitted status`);
            }
        }
        const updateQuery = `
      UPDATE time_entries 
      SET status = 'Approved', approved_by = ?, approved_at = NOW()
      WHERE id IN (${entryIds.map(() => '?').join(',')})
    `;
        await pool.query(updateQuery, [approverId, ...entryIds]);
    }
    async rejectTimeEntries(entryIds, approverId, reason) {
        const pool = await getPool();
        // Verify all entries are in Submitted status
        const checkQuery = `
      SELECT id, status 
      FROM time_entries 
      WHERE id IN (${entryIds.map(() => '?').join(',')})
    `;
        const [entries] = await pool.query(checkQuery, entryIds);
        for (const entry of entries) {
            if (entry.status !== 'Submitted') {
                throw new Error(`Entry ${entry.id} is not in Submitted status`);
            }
        }
        const updateQuery = `
      UPDATE time_entries 
      SET status = 'Rejected', approved_by = ?, approved_at = NOW(), rejection_reason = ?
      WHERE id IN (${entryIds.map(() => '?').join(',')})
    `;
        await pool.query(updateQuery, [approverId, reason, ...entryIds]);
    }
    // ================================================
    // Timers
    // ================================================
    async getRunningTimer(userId) {
        const pool = await getPool();
        const query = `
      SELECT 
        rt.*,
        m.matter_number,
        m.title as matter_title,
        ac.name as activity_name,
        TIMESTAMPDIFF(SECOND, rt.start_time, NOW()) as elapsed_seconds
      FROM running_timers rt
      JOIN matters m ON rt.matter_id = m.id
      LEFT JOIN activity_codes ac ON rt.activity_code = ac.code
      WHERE rt.user_id = ?
    `;
        const [rows] = await pool.query(query, [userId]);
        if (rows.length === 0) {
            return null;
        }
        const timer = rows[0];
        const hours = Math.floor(timer.elapsed_seconds / 3600);
        const minutes = Math.floor((timer.elapsed_seconds % 3600) / 60);
        const seconds = timer.elapsed_seconds % 60;
        return {
            ...timer,
            elapsed_formatted: `${hours}:${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`,
        };
    }
    async startTimer(data, userId) {
        const pool = await getPool();
        // Check if user already has a running timer
        const existing = await this.getRunningTimer(userId);
        if (existing) {
            throw new Error('You already have a running timer. Stop it before starting a new one.');
        }
        const query = `
      INSERT INTO running_timers (user_id, matter_id, activity_code, description, start_time)
      VALUES (?, ?, ?, ?, NOW())
    `;
        await pool.query(query, [
            userId,
            data.matter_id,
            data.activity_code,
            data.description || null,
        ]);
        return this.getRunningTimer(userId);
    }
    async stopTimer(userId) {
        const pool = await getPool();
        // Get running timer
        const timer = await this.getRunningTimer(userId);
        if (!timer) {
            throw new Error('No running timer found');
        }
        // Calculate hours
        const hours = Math.round((timer.elapsed_seconds / 3600) * 100) / 100; // Round to 2 decimals
        // Create time entry
        const createQuery = `
      INSERT INTO time_entries (
        user_id, matter_id, entry_date, hours, activity_code, 
        description, billable, created_by
      ) VALUES (?, ?, CURDATE(), ?, ?, ?, TRUE, ?)
    `;
        const [result] = await pool.query(createQuery, [
            userId,
            timer.matter_id,
            hours,
            timer.activity_code,
            timer.description || 'Timer entry',
            userId,
        ]);
        // Delete timer
        await pool.query('DELETE FROM running_timers WHERE user_id = ?', [userId]);
        return this.getTimeEntryById(result.insertId, userId, true);
    }
    // ================================================
    // Activity Codes
    // ================================================
    async listActivityCodes(activeOnly = true) {
        const pool = await getPool();
        const whereClause = activeOnly ? 'WHERE is_active = TRUE' : '';
        const query = `
      SELECT * FROM activity_codes 
      ${whereClause}
      ORDER BY sort_order, name
    `;
        const [rows] = await pool.query(query);
        return rows;
    }
    async getActivityCodeById(id) {
        const pool = await getPool();
        const [rows] = await pool.query('SELECT * FROM activity_codes WHERE id = ?', [id]);
        return rows.length > 0 ? rows[0] : null;
    }
    async createActivityCode(data) {
        const pool = await getPool();
        const query = `
      INSERT INTO activity_codes (code, name, description, default_billable, sort_order)
      VALUES (?, ?, ?, ?, ?)
    `;
        const [result] = await pool.query(query, [
            data.code,
            data.name,
            data.description || null,
            data.default_billable,
            data.sort_order,
        ]);
        return this.getActivityCodeById(result.insertId);
    }
    async updateActivityCode(id, data) {
        const pool = await getPool();
        const updates = [];
        const params = [];
        if (data.name !== undefined) {
            updates.push('name = ?');
            params.push(data.name);
        }
        if (data.description !== undefined) {
            updates.push('description = ?');
            params.push(data.description);
        }
        if (data.default_billable !== undefined) {
            updates.push('default_billable = ?');
            params.push(data.default_billable);
        }
        if (data.is_active !== undefined) {
            updates.push('is_active = ?');
            params.push(data.is_active);
        }
        if (data.sort_order !== undefined) {
            updates.push('sort_order = ?');
            params.push(data.sort_order);
        }
        if (updates.length === 0) {
            return this.getActivityCodeById(id);
        }
        const query = `UPDATE activity_codes SET ${updates.join(', ')} WHERE id = ?`;
        await pool.query(query, [...params, id]);
        return this.getActivityCodeById(id);
    }
}
export const timeRepo = new TimeRepository();
//# sourceMappingURL=repo.js.map