import { z } from 'zod';
// ================================================
// Trust Account DTOs
// ================================================
// Create Trust Account
export const createTrustAccountSchema = z.object({
    client_id: z.number().int().positive(),
    matter_id: z.number().int().positive().optional(),
    account_name: z.string().min(1).max(200),
    account_number: z.string().max(50).optional(),
    bank_name: z.string().max(100).optional(),
    currency: z.string().length(3).default('GHS'),
    account_type: z.enum(['Client Trust', 'Matter Trust', 'General Trust']).default('Client Trust'),
    notes: z.string().max(2000).optional(),
});
// Update Trust Account
export const updateTrustAccountSchema = z.object({
    account_name: z.string().min(1).max(200).optional(),
    account_number: z.string().max(50).optional(),
    bank_name: z.string().max(100).optional(),
    notes: z.string().max(2000).optional(),
    is_active: z.boolean().optional(),
});
// ================================================
// Trust Transaction DTOs
// ================================================
// Create Trust Transaction (Deposit/Withdrawal)
export const createTrustTransactionSchema = z.object({
    trust_account_id: z.number().int().positive(),
    transaction_type: z.enum(['Deposit', 'Withdrawal', 'Transfer Out', 'Transfer In', 'Fee Transfer', 'Refund']),
    amount: z.number().positive(),
    transaction_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
    reference_number: z.string().max(100).optional(),
    description: z.string().min(1).max(5000),
    payment_method: z.string().max(50).optional(),
    payee: z.string().max(200).optional(),
    related_invoice_id: z.number().int().positive().optional(),
    related_matter_id: z.number().int().positive().optional(),
    requires_approval: z.boolean().default(true),
});
// Transfer Between Trust Accounts
export const transferBetweenTrustSchema = z.object({
    from_account_id: z.number().int().positive(),
    to_account_id: z.number().int().positive(),
    amount: z.number().positive(),
    transaction_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
    description: z.string().min(1).max(5000),
    reference_number: z.string().max(100).optional(),
});
// Approve Trust Transaction
export const approveTrustTransactionSchema = z.object({
    approval_notes: z.string().max(1000).optional(),
});
// Reject Trust Transaction
export const rejectTrustTransactionSchema = z.object({
    rejection_reason: z.string().min(1).max(1000),
});
// Query Trust Transactions
export const queryTrustTransactionsSchema = z.object({
    page: z.string().optional().transform(val => val ? parseInt(val) : 1),
    limit: z.string().optional().transform(val => val ? parseInt(val) : 50),
    trust_account_id: z.string().optional().transform(val => val ? parseInt(val) : undefined),
    transaction_type: z.enum(['Deposit', 'Withdrawal', 'Transfer Out', 'Transfer In', 'Fee Transfer', 'Refund']).optional(),
    status: z.enum(['Pending', 'Approved', 'Rejected', 'Posted']).optional(),
    start_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
    end_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/).optional(),
});
// ================================================
// Trust Reconciliation DTOs
// ================================================
// Create Trust Reconciliation
export const createTrustReconciliationSchema = z.object({
    trust_account_id: z.number().int().positive(),
    reconciliation_date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/),
    bank_statement_balance: z.number(),
    book_balance: z.number(),
    reconciled_balance: z.number(),
    unreconciled_items: z.string().max(5000).optional(),
    notes: z.string().max(2000).optional(),
});
// Update Trust Reconciliation
export const updateTrustReconciliationSchema = z.object({
    bank_statement_balance: z.number().optional(),
    reconciled_balance: z.number().optional(),
    unreconciled_items: z.string().max(5000).optional(),
    notes: z.string().max(2000).optional(),
    status: z.enum(['Draft', 'Completed', 'Reviewed']).optional(),
});
//# sourceMappingURL=dto.js.map