import { Router } from 'express';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { trustRepo } from './repo.js';
import { createTrustAccountSchema, updateTrustAccountSchema, createTrustTransactionSchema, transferBetweenTrustSchema, approveTrustTransactionSchema, rejectTrustTransactionSchema, queryTrustTransactionsSchema, createTrustReconciliationSchema, updateTrustReconciliationSchema, } from './dto.js';
const router = Router();
// ================================================
// Trust Accounts
// ================================================
/**
 * GET /api/trust/accounts
 * List trust accounts
 */
router.get('/accounts', authenticate, requirePermission('trust:read'), async (req, res, next) => {
    try {
        const clientId = req.query.client_id ? parseInt(req.query.client_id) : undefined;
        const isActive = req.query.is_active === 'false' ? false : req.query.is_active === 'true' ? true : undefined;
        const accounts = await trustRepo.listTrustAccounts(clientId, isActive);
        res.json({ success: true, data: { accounts } });
    }
    catch (error) {
        next(error);
    }
});
/**
 * GET /api/trust/accounts/:id
 * Get single trust account
 */
router.get('/accounts/:id', authenticate, requirePermission('trust:read'), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const account = await trustRepo.getTrustAccountById(id);
        if (!account) {
            res.status(404).json({ success: false, error: 'Trust account not found' });
            return;
        }
        res.json({ success: true, data: account });
    }
    catch (error) {
        next(error);
    }
});
/**
 * POST /api/trust/accounts
 * Create trust account
 */
router.post('/accounts', authenticate, requirePermission('trust:manage'), async (req, res, next) => {
    try {
        const data = createTrustAccountSchema.parse(req.body);
        const account = await trustRepo.createTrustAccount(data, req.user.id);
        res.status(201).json({ success: true, data: account });
    }
    catch (error) {
        next(error);
    }
});
/**
 * PUT /api/trust/accounts/:id
 * Update trust account
 */
router.put('/accounts/:id', authenticate, requirePermission('trust:manage'), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const data = updateTrustAccountSchema.parse(req.body);
        const account = await trustRepo.updateTrustAccount(id, data);
        res.json({ success: true, data: account });
    }
    catch (error) {
        next(error);
    }
});
// ================================================
// Trust Transactions
// ================================================
/**
 * GET /api/trust/transactions
 * List trust transactions with filters
 */
router.get('/transactions', authenticate, requirePermission('trust:read'), async (req, res, next) => {
    try {
        const query = queryTrustTransactionsSchema.parse(req.query);
        const result = await trustRepo.listTrustTransactions(query);
        res.json({ success: true, data: result });
    }
    catch (error) {
        next(error);
    }
});
/**
 * GET /api/trust/transactions/:id
 * Get single trust transaction
 */
router.get('/transactions/:id', authenticate, requirePermission('trust:read'), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const transaction = await trustRepo.getTrustTransactionById(id);
        if (!transaction) {
            res.status(404).json({ success: false, error: 'Trust transaction not found' });
            return;
        }
        res.json({ success: true, data: transaction });
    }
    catch (error) {
        next(error);
    }
});
/**
 * POST /api/trust/transactions
 * Create trust transaction (deposit/withdrawal)
 */
router.post('/transactions', authenticate, requirePermission('trust:transact'), async (req, res, next) => {
    try {
        const data = createTrustTransactionSchema.parse(req.body);
        const transaction = await trustRepo.createTrustTransaction(data, req.user.id);
        res.status(201).json({
            success: true,
            data: transaction,
            message: data.requires_approval ? 'Transaction created and pending approval' : 'Transaction posted'
        });
    }
    catch (error) {
        next(error);
    }
});
/**
 * POST /api/trust/transactions/transfer
 * Transfer between trust accounts
 */
router.post('/transactions/transfer', authenticate, requirePermission('trust:transact'), async (req, res, next) => {
    try {
        const data = transferBetweenTrustSchema.parse(req.body);
        const result = await trustRepo.transferBetweenTrust(data, req.user.id);
        res.status(201).json({
            success: true,
            data: result,
            message: 'Transfer created and pending dual approval'
        });
    }
    catch (error) {
        next(error);
    }
});
/**
 * POST /api/trust/transactions/:id/approve
 * Approve trust transaction (dual control)
 */
router.post('/transactions/:id/approve', authenticate, requirePermission('trust:approve'), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const data = approveTrustTransactionSchema.parse(req.body);
        const transaction = await trustRepo.approveTrustTransaction(id, req.user.id, data);
        res.json({
            success: true,
            data: transaction,
            message: 'Transaction approved and posted'
        });
    }
    catch (error) {
        next(error);
    }
});
/**
 * POST /api/trust/transactions/:id/reject
 * Reject trust transaction (dual control)
 */
router.post('/transactions/:id/reject', authenticate, requirePermission('trust:approve'), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const data = rejectTrustTransactionSchema.parse(req.body);
        const transaction = await trustRepo.rejectTrustTransaction(id, req.user.id, data);
        res.json({
            success: true,
            data: transaction,
            message: 'Transaction rejected'
        });
    }
    catch (error) {
        next(error);
    }
});
// ================================================
// Trust Reconciliations
// ================================================
/**
 * GET /api/trust/reconciliations
 * List trust reconciliations
 */
router.get('/reconciliations', authenticate, requirePermission('trust:read'), async (req, res, next) => {
    try {
        const trustAccountId = req.query.trust_account_id ? parseInt(req.query.trust_account_id) : undefined;
        const reconciliations = await trustRepo.listTrustReconciliations(trustAccountId);
        res.json({ success: true, data: { reconciliations } });
    }
    catch (error) {
        next(error);
    }
});
/**
 * GET /api/trust/reconciliations/:id
 * Get single trust reconciliation
 */
router.get('/reconciliations/:id', authenticate, requirePermission('trust:read'), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const reconciliation = await trustRepo.getTrustReconciliationById(id);
        if (!reconciliation) {
            res.status(404).json({ success: false, error: 'Trust reconciliation not found' });
            return;
        }
        res.json({ success: true, data: reconciliation });
    }
    catch (error) {
        next(error);
    }
});
/**
 * POST /api/trust/reconciliations
 * Create trust reconciliation
 */
router.post('/reconciliations', authenticate, requirePermission('trust:manage'), async (req, res, next) => {
    try {
        const data = createTrustReconciliationSchema.parse(req.body);
        const reconciliation = await trustRepo.createTrustReconciliation(data, req.user.id);
        res.status(201).json({ success: true, data: reconciliation });
    }
    catch (error) {
        next(error);
    }
});
/**
 * PUT /api/trust/reconciliations/:id
 * Update trust reconciliation
 */
router.put('/reconciliations/:id', authenticate, requirePermission('trust:manage'), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const data = updateTrustReconciliationSchema.parse(req.body);
        const reconciliation = await trustRepo.updateTrustReconciliation(id, data);
        res.json({ success: true, data: reconciliation });
    }
    catch (error) {
        next(error);
    }
});
/**
 * POST /api/trust/reconciliations/:id/review
 * Review trust reconciliation (dual control)
 */
router.post('/reconciliations/:id/review', authenticate, requirePermission('trust:approve'), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const reconciliation = await trustRepo.reviewTrustReconciliation(id, req.user.id);
        res.json({
            success: true,
            data: reconciliation,
            message: 'Reconciliation reviewed and approved'
        });
    }
    catch (error) {
        next(error);
    }
});
/**
 * GET /api/trust/summary
 * Get trust accounts summary
 */
router.get('/summary', authenticate, requirePermission('trust:read'), async (req, res, next) => {
    try {
        const summary = await trustRepo.getTrustSummary();
        res.json({ success: true, data: summary });
    }
    catch (error) {
        next(error);
    }
});
export default router;
//# sourceMappingURL=routes.js.map