import { getPool } from '../../db/pool.js';
import { hashPassword } from '../../utils/crypto.js';
export class UserRepository {
    /**
     * Get all users with employee, role, and department info
     */
    async findAll() {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT 
        u.id, u.username, u.employee_id,
        CONCAT(e.first_name, ' ', e.last_name) as employee_name,
        u.role_id, r.name as role_name,
        e.department_id, d.name as department_name,
        u.created_at
       FROM users u
       JOIN employees e ON e.id = u.employee_id
       JOIN roles r ON r.id = u.role_id
       JOIN departments d ON d.id = e.department_id
       ORDER BY u.username ASC`);
        return rows;
    }
    /**
     * Get user by ID
     */
    async findById(id) {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT 
        u.id, u.username, u.employee_id,
        CONCAT(e.first_name, ' ', e.last_name) as employee_name,
        u.role_id, r.name as role_name,
        e.department_id, d.name as department_name,
        u.created_at
       FROM users u
       JOIN employees e ON e.id = u.employee_id
       JOIN roles r ON r.id = u.role_id
       JOIN departments d ON d.id = e.department_id
       WHERE u.id = ?`, [id]);
        return rows.length > 0 ? rows[0] : null;
    }
    /**
     * Check if username exists (excluding specific ID for updates)
     */
    async usernameExists(username, excludeId) {
        const pool = await getPool();
        const query = excludeId
            ? 'SELECT COUNT(*) as count FROM users WHERE username = ? AND id != ?'
            : 'SELECT COUNT(*) as count FROM users WHERE username = ?';
        const params = excludeId ? [username, excludeId] : [username];
        const [rows] = await pool.query(query, params);
        return rows[0].count > 0;
    }
    /**
     * Check if employee already has a user account
     */
    async employeeHasUser(employeeId, excludeUserId) {
        const pool = await getPool();
        const query = excludeUserId
            ? 'SELECT COUNT(*) as count FROM users WHERE employee_id = ? AND id != ?'
            : 'SELECT COUNT(*) as count FROM users WHERE employee_id = ?';
        const params = excludeUserId ? [employeeId, excludeUserId] : [employeeId];
        const [rows] = await pool.query(query, params);
        return rows[0].count > 0;
    }
    /**
     * Check if employee exists
     */
    async employeeExists(employeeId) {
        const pool = await getPool();
        const [rows] = await pool.query('SELECT COUNT(*) as count FROM employees WHERE id = ?', [employeeId]);
        return rows[0].count > 0;
    }
    /**
     * Check if role exists
     */
    async roleExists(roleId) {
        const pool = await getPool();
        const [rows] = await pool.query('SELECT COUNT(*) as count FROM roles WHERE id = ?', [roleId]);
        return rows[0].count > 0;
    }
    /**
     * Create user
     */
    async create(data) {
        const pool = await getPool();
        const passwordHash = await hashPassword(data.password);
        const [result] = await pool.query(`INSERT INTO users (username, password_hash, employee_id, role_id)
       VALUES (?, ?, ?, ?)`, [data.username, passwordHash, data.employee_id, data.role_id]);
        const newUser = await this.findById(result.insertId);
        if (!newUser) {
            throw new Error('Failed to create user');
        }
        return newUser;
    }
    /**
     * Update user
     */
    async update(id, data) {
        const pool = await getPool();
        const updates = [];
        const values = [];
        if (data.username !== undefined) {
            updates.push('username = ?');
            values.push(data.username);
        }
        if (data.password !== undefined) {
            const passwordHash = await hashPassword(data.password);
            updates.push('password_hash = ?');
            values.push(passwordHash);
        }
        if (data.employee_id !== undefined) {
            updates.push('employee_id = ?');
            values.push(data.employee_id);
        }
        if (data.role_id !== undefined) {
            updates.push('role_id = ?');
            values.push(data.role_id);
        }
        if (updates.length === 0) {
            const user = await this.findById(id);
            if (!user) {
                throw new Error('User not found');
            }
            return user;
        }
        values.push(id);
        await pool.query(`UPDATE users SET ${updates.join(', ')} WHERE id = ?`, values);
        const updated = await this.findById(id);
        if (!updated) {
            throw new Error('User not found');
        }
        return updated;
    }
    /**
     * Delete user
     */
    async delete(id) {
        const pool = await getPool();
        // Delete associated sessions first
        await pool.query('DELETE FROM sessions WHERE user_id = ?', [id]);
        // Delete user
        await pool.query('DELETE FROM users WHERE id = ?', [id]);
    }
    /**
     * Log user action to audit log
     */
    async logAction(userId, targetUserId, action, meta) {
        const pool = await getPool();
        await pool.query(`INSERT INTO audit_logs (user_id, entity, entity_id, action, meta)
       VALUES (?, 'user', ?, ?, ?)`, [userId, targetUserId, action, JSON.stringify(meta)]);
    }
}
//# sourceMappingURL=repo.js.map