import { Router } from 'express';
import { UserRepository } from './repo.js';
import { createUserSchema, updateUserSchema, deleteUserSchema, getUserSchema, } from './dto.js';
import { validate } from '../../middleware/validate.js';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { AppError } from '../../middleware/errors.js';
const router = Router();
const userRepo = new UserRepository();
// All routes require authentication
router.use(authenticate);
/**
 * GET /api/users
 * List all users (Admin/Manager only)
 */
router.get('/', requirePermission('users:read'), async (req, res, next) => {
    try {
        const users = await userRepo.findAll();
        res.json({
            success: true,
            data: { users },
        });
    }
    catch (error) {
        next(error);
    }
});
/**
 * GET /api/users/:id
 * Get user by ID (Admin/Manager only)
 */
router.get('/:id', requirePermission('users:read'), validate(getUserSchema), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const user = await userRepo.findById(id);
        if (!user) {
            throw new AppError(404, 'User not found');
        }
        res.json({
            success: true,
            data: { user },
        });
    }
    catch (error) {
        next(error);
    }
});
/**
 * POST /api/users
 * Create new user (Admin only)
 */
router.post('/', requirePermission('users:create'), validate(createUserSchema), async (req, res, next) => {
    try {
        const { username, password, employee_id, role_id } = req.body;
        // Check if username exists
        const usernameExists = await userRepo.usernameExists(username);
        if (usernameExists) {
            throw new AppError(409, 'Username already exists');
        }
        // Check if employee exists
        const employeeExists = await userRepo.employeeExists(employee_id);
        if (!employeeExists) {
            throw new AppError(404, 'Employee not found');
        }
        // Check if employee already has a user account
        const hasUser = await userRepo.employeeHasUser(employee_id);
        if (hasUser) {
            throw new AppError(409, 'Employee already has a user account');
        }
        // Check if role exists
        const roleExists = await userRepo.roleExists(role_id);
        if (!roleExists) {
            throw new AppError(404, 'Role not found');
        }
        const user = await userRepo.create({
            username,
            password,
            employee_id,
            role_id,
        });
        // Log action
        await userRepo.logAction(req.user.id, user.id, 'CREATE', {
            username: user.username,
            employee_id: user.employee_id,
            role_id: user.role_id,
        });
        res.status(201).json({
            success: true,
            data: { user },
        });
    }
    catch (error) {
        next(error);
    }
});
/**
 * PUT /api/users/:id
 * Update user (Admin only)
 */
router.put('/:id', requirePermission('users:update'), validate(updateUserSchema), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        const { username, password, employee_id, role_id } = req.body;
        // Check if user exists
        const existing = await userRepo.findById(id);
        if (!existing) {
            throw new AppError(404, 'User not found');
        }
        // Check for duplicate username (if username is being changed)
        if (username && username !== existing.username) {
            const usernameExists = await userRepo.usernameExists(username, id);
            if (usernameExists) {
                throw new AppError(409, 'Username already exists');
            }
        }
        // If employee is changing, check if it exists and doesn't have another user
        if (employee_id && employee_id !== existing.employee_id) {
            const employeeExists = await userRepo.employeeExists(employee_id);
            if (!employeeExists) {
                throw new AppError(404, 'Employee not found');
            }
            const hasUser = await userRepo.employeeHasUser(employee_id, id);
            if (hasUser) {
                throw new AppError(409, 'Employee already has a user account');
            }
        }
        // If role is changing, check if it exists
        if (role_id && role_id !== existing.role_id) {
            const roleExists = await userRepo.roleExists(role_id);
            if (!roleExists) {
                throw new AppError(404, 'Role not found');
            }
        }
        const user = await userRepo.update(id, {
            username,
            password,
            employee_id,
            role_id,
        });
        // Log action
        await userRepo.logAction(req.user.id, id, 'UPDATE', {
            old: existing,
            new: user,
        });
        res.json({
            success: true,
            data: { user },
        });
    }
    catch (error) {
        next(error);
    }
});
/**
 * DELETE /api/users/:id
 * Delete user (Admin only)
 */
router.delete('/:id', requirePermission('users:delete'), validate(deleteUserSchema), async (req, res, next) => {
    try {
        const id = parseInt(req.params.id);
        // Check if user exists
        const user = await userRepo.findById(id);
        if (!user) {
            throw new AppError(404, 'User not found');
        }
        // Prevent deleting yourself
        if (id === req.user.id) {
            throw new AppError(400, 'Cannot delete your own user account');
        }
        await userRepo.delete(id);
        // Log action
        await userRepo.logAction(req.user.id, id, 'DELETE', {
            username: user.username,
            employee_id: user.employee_id,
        });
        res.json({
            success: true,
            data: { message: 'User deleted successfully' },
        });
    }
    catch (error) {
        next(error);
    }
});
export default router;
//# sourceMappingURL=routes.js.map