import { getPool } from '../../db/pool.js';
export class WIPRepository {
    // ================================================
    // WIP Summary
    // ================================================
    async refreshWIPSummary(matterId) {
        const pool = await getPool();
        // Delete existing summaries for the matter(s)
        if (matterId) {
            await pool.query('DELETE FROM wip_summary WHERE matter_id = ?', [matterId]);
        }
        else {
            await pool.query('DELETE FROM wip_summary');
        }
        // Recalculate WIP for all matters with unbilled items
        const query = `
      INSERT INTO wip_summary (
        matter_id, unbilled_time_hours, unbilled_time_value,
        unbilled_expenses, total_wip, last_invoice_date, oldest_unbilled_date
      )
      SELECT 
        m.id as matter_id,
        COALESCE(SUM(CASE WHEN te.id IS NOT NULL THEN te.hours ELSE 0 END), 0) as unbilled_time_hours,
        COALESCE(SUM(CASE WHEN te.id IS NOT NULL THEN te.total_amount ELSE 0 END), 0) as unbilled_time_value,
        COALESCE(SUM(CASE WHEN ee.id IS NOT NULL THEN ee.billable_amount ELSE 0 END), 0) as unbilled_expenses,
        COALESCE(SUM(CASE WHEN te.id IS NOT NULL THEN te.total_amount ELSE 0 END), 0) + 
          COALESCE(SUM(CASE WHEN ee.id IS NOT NULL THEN ee.billable_amount ELSE 0 END), 0) as total_wip,
        (SELECT MAX(i.invoice_date) FROM invoices i WHERE i.matter_id = m.id) as last_invoice_date,
        LEAST(
          COALESCE((SELECT MIN(te2.entry_date) FROM time_entries te2 
                    WHERE te2.matter_id = m.id AND te2.status = 'Approved' AND te2.billable = TRUE), '9999-12-31'),
          COALESCE((SELECT MIN(ee2.expense_date) FROM expense_entries ee2 
                    WHERE ee2.matter_id = m.id AND ee2.status = 'Approved' AND ee2.billable = TRUE), '9999-12-31')
        ) as oldest_unbilled_date
      FROM matters m
      LEFT JOIN time_entries te ON m.id = te.matter_id 
        AND te.status = 'Approved' 
        AND te.billable = TRUE
      LEFT JOIN expense_entries ee ON m.id = ee.matter_id 
        AND ee.status = 'Approved' 
        AND ee.billable = TRUE
      ${matterId ? 'WHERE m.id = ?' : ''}
      GROUP BY m.id
      HAVING total_wip > 0
    `;
        await pool.query(query, matterId ? [matterId] : []);
    }
    async listWIPSummary(query) {
        const pool = await getPool();
        const { matter_id, client_id, min_balance } = query;
        // Refresh WIP before querying
        await this.refreshWIPSummary(matter_id);
        let whereConditions = [];
        let params = [];
        if (matter_id) {
            whereConditions.push('ws.matter_id = ?');
            params.push(matter_id);
        }
        if (client_id) {
            whereConditions.push('m.client_id = ?');
            params.push(client_id);
        }
        if (min_balance) {
            whereConditions.push('ws.total_wip >= ?');
            params.push(min_balance);
        }
        const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';
        const sql = `
      SELECT 
        ws.*,
        m.matter_number,
        m.title as matter_title,
        m.client_id,
        c.name as client_name
      FROM wip_summary ws
      JOIN matters m ON ws.matter_id = m.id
      JOIN clients c ON m.client_id = c.id
      ${whereClause}
      ORDER BY ws.total_wip DESC
    `;
        const [rows] = await pool.query(sql, params);
        return rows;
    }
    async getWIPAnalytics() {
        const pool = await getPool();
        // Refresh all WIP
        await this.refreshWIPSummary();
        // Get total WIP metrics
        const totalsQuery = `
      SELECT 
        COALESCE(SUM(total_wip), 0) as total_wip_value,
        COALESCE(SUM(unbilled_time_value), 0) as total_unbilled_time,
        COALESCE(SUM(unbilled_expenses), 0) as total_unbilled_expenses,
        COUNT(*) as matters_with_wip,
        COALESCE(AVG(DATEDIFF(CURDATE(), oldest_unbilled_date)), 0) as avg_days_unbilled
      FROM wip_summary
    `;
        const [totalsRows] = await pool.query(totalsQuery);
        const analytics = totalsRows[0];
        // Get top WIP matters
        const topMattersQuery = `
      SELECT 
        ws.matter_id,
        m.matter_number,
        m.title as matter_title,
        c.name as client_name,
        ws.total_wip as wip_value
      FROM wip_summary ws
      JOIN matters m ON ws.matter_id = m.id
      JOIN clients c ON m.client_id = c.id
      ORDER BY ws.total_wip DESC
      LIMIT 10
    `;
        const [topMattersRows] = await pool.query(topMattersQuery);
        analytics.top_wip_matters = topMattersRows;
        // Get WIP by client
        const byClientQuery = `
      SELECT 
        c.id as client_id,
        c.name as client_name,
        COALESCE(SUM(ws.total_wip), 0) as wip_value,
        COUNT(DISTINCT ws.matter_id) as matter_count
      FROM clients c
      JOIN matters m ON c.id = m.client_id
      JOIN wip_summary ws ON m.id = ws.matter_id
      GROUP BY c.id, c.name
      ORDER BY wip_value DESC
      LIMIT 10
    `;
        const [byClientRows] = await pool.query(byClientQuery);
        analytics.wip_by_client = byClientRows;
        return analytics;
    }
    // ================================================
    // Revenue Recognition
    // ================================================
    async listRevenueRecognition(query) {
        const pool = await getPool();
        const { page = 1, limit = 50, matter_id, client_id, start_month, end_month } = query;
        const offset = (page - 1) * limit;
        let whereConditions = [];
        let params = [];
        if (matter_id) {
            whereConditions.push('rr.matter_id = ?');
            params.push(matter_id);
        }
        if (client_id) {
            whereConditions.push('m.client_id = ?');
            params.push(client_id);
        }
        if (start_month) {
            whereConditions.push('rr.recognition_month >= ?');
            params.push(start_month);
        }
        if (end_month) {
            whereConditions.push('rr.recognition_month <= ?');
            params.push(end_month);
        }
        const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';
        // Get total count
        const countQuery = `SELECT COUNT(*) as total FROM revenue_recognition rr JOIN matters m ON rr.matter_id = m.id ${whereClause}`;
        const [countRows] = await pool.query(countQuery, params);
        const total = countRows[0].total;
        // Get revenue records
        const sql = `
      SELECT 
        rr.*,
        m.matter_number,
        m.title as matter_title,
        m.client_id,
        c.name as client_name,
        u.username as created_by_name,
        (rr.total_value - rr.invoiced_amount) as unbilled_value,
        (rr.invoiced_amount - rr.collected_amount) as uncollected_value
      FROM revenue_recognition rr
      JOIN matters m ON rr.matter_id = m.id
      JOIN clients c ON m.client_id = c.id
      JOIN users u ON rr.created_by = u.id
      ${whereClause}
      ORDER BY rr.recognition_month DESC, rr.created_at DESC
      LIMIT ? OFFSET ?
    `;
        const [rows] = await pool.query(sql, [...params, limit, offset]);
        return {
            revenue: rows,
            pagination: {
                page,
                limit,
                total,
                totalPages: Math.ceil(total / limit),
            },
        };
    }
    async getRevenueRecognitionById(id) {
        const pool = await getPool();
        const query = `
      SELECT 
        rr.*,
        m.matter_number,
        m.title as matter_title,
        m.client_id,
        c.name as client_name,
        u.username as created_by_name,
        (rr.total_value - rr.invoiced_amount) as unbilled_value,
        (rr.invoiced_amount - rr.collected_amount) as uncollected_value
      FROM revenue_recognition rr
      JOIN matters m ON rr.matter_id = m.id
      JOIN clients c ON m.client_id = c.id
      JOIN users u ON rr.created_by = u.id
      WHERE rr.id = ?
    `;
        const [rows] = await pool.query(query, [id]);
        return rows.length > 0 ? rows[0] : null;
    }
    async createRevenueRecognition(data, userId) {
        const pool = await getPool();
        const query = `
      INSERT INTO revenue_recognition (
        matter_id, recognition_month, time_value, expense_value,
        total_value, invoiced_amount, collected_amount, notes, created_by
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
    `;
        const [result] = await pool.query(query, [
            data.matter_id,
            data.recognition_month,
            data.time_value,
            data.expense_value,
            data.total_value,
            data.invoiced_amount,
            data.collected_amount,
            data.notes || null,
            userId,
        ]);
        return this.getRevenueRecognitionById(result.insertId);
    }
    async updateRevenueRecognition(id, data) {
        const pool = await getPool();
        const existing = await this.getRevenueRecognitionById(id);
        if (!existing) {
            throw new Error('Revenue recognition record not found');
        }
        const updates = [];
        const params = [];
        if (data.time_value !== undefined) {
            updates.push('time_value = ?');
            params.push(data.time_value);
        }
        if (data.expense_value !== undefined) {
            updates.push('expense_value = ?');
            params.push(data.expense_value);
        }
        if (data.total_value !== undefined) {
            updates.push('total_value = ?');
            params.push(data.total_value);
        }
        if (data.invoiced_amount !== undefined) {
            updates.push('invoiced_amount = ?');
            params.push(data.invoiced_amount);
        }
        if (data.collected_amount !== undefined) {
            updates.push('collected_amount = ?');
            params.push(data.collected_amount);
        }
        if (data.notes !== undefined) {
            updates.push('notes = ?');
            params.push(data.notes);
        }
        if (updates.length === 0) {
            return existing;
        }
        const query = `UPDATE revenue_recognition SET ${updates.join(', ')} WHERE id = ?`;
        await pool.query(query, [...params, id]);
        return this.getRevenueRecognitionById(id);
    }
    async getRevenueAnalytics(startMonth, endMonth) {
        const pool = await getPool();
        let whereConditions = [];
        let params = [];
        if (startMonth) {
            whereConditions.push('recognition_month >= ?');
            params.push(startMonth);
        }
        if (endMonth) {
            whereConditions.push('recognition_month <= ?');
            params.push(endMonth);
        }
        const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';
        // Get total metrics
        const totalsQuery = `
      SELECT 
        COALESCE(SUM(total_value), 0) as total_recognized,
        COALESCE(SUM(invoiced_amount), 0) as total_invoiced,
        COALESCE(SUM(collected_amount), 0) as total_collected,
        COALESCE(SUM(total_value - invoiced_amount), 0) as total_unbilled,
        COALESCE(SUM(invoiced_amount - collected_amount), 0) as total_uncollected
      FROM revenue_recognition
      ${whereClause}
    `;
        const [totalsRows] = await pool.query(totalsQuery, params);
        const analytics = totalsRows[0];
        // Calculate rates
        analytics.collection_rate = analytics.total_invoiced > 0
            ? (analytics.total_collected / analytics.total_invoiced) * 100
            : 0;
        analytics.realization_rate = analytics.total_recognized > 0
            ? (analytics.total_invoiced / analytics.total_recognized) * 100
            : 0;
        // Get by month breakdown
        const monthlyQuery = `
      SELECT 
        recognition_month as month,
        COALESCE(SUM(total_value), 0) as recognized,
        COALESCE(SUM(invoiced_amount), 0) as invoiced,
        COALESCE(SUM(collected_amount), 0) as collected
      FROM revenue_recognition
      ${whereClause}
      GROUP BY recognition_month
      ORDER BY recognition_month DESC
      LIMIT 12
    `;
        const [monthlyRows] = await pool.query(monthlyQuery, params);
        analytics.by_month = monthlyRows;
        return analytics;
    }
    async autoGenerateRevenueForMonth(month, userId) {
        const pool = await getPool();
        // Get all matters with activity in the month
        const startDate = `${month}-01`;
        const endDate = `${month}-31`;
        const query = `
      INSERT INTO revenue_recognition (
        matter_id, recognition_month, time_value, expense_value,
        total_value, invoiced_amount, collected_amount, created_by
      )
      SELECT 
        m.id as matter_id,
        ? as recognition_month,
        COALESCE(SUM(CASE 
          WHEN te.entry_date BETWEEN ? AND ? AND te.status IN ('Approved', 'Invoiced')
          THEN te.total_amount ELSE 0 END), 0) as time_value,
        COALESCE(SUM(CASE 
          WHEN ee.expense_date BETWEEN ? AND ? AND ee.status IN ('Approved', 'Invoiced')
          THEN ee.billable_amount ELSE 0 END), 0) as expense_value,
        COALESCE(SUM(CASE 
          WHEN te.entry_date BETWEEN ? AND ? AND te.status IN ('Approved', 'Invoiced')
          THEN te.total_amount ELSE 0 END), 0) +
        COALESCE(SUM(CASE 
          WHEN ee.expense_date BETWEEN ? AND ? AND ee.status IN ('Approved', 'Invoiced')
          THEN ee.billable_amount ELSE 0 END), 0) as total_value,
        COALESCE(SUM(CASE 
          WHEN i.invoice_date BETWEEN ? AND ?
          THEN i.total_amount ELSE 0 END), 0) as invoiced_amount,
        COALESCE(SUM(CASE 
          WHEN r.receipt_date BETWEEN ? AND ?
          THEN pa.amount ELSE 0 END), 0) as collected_amount,
        ? as created_by
      FROM matters m
      LEFT JOIN time_entries te ON m.id = te.matter_id
      LEFT JOIN expense_entries ee ON m.id = ee.matter_id
      LEFT JOIN invoices i ON m.id = i.matter_id
      LEFT JOIN payment_allocations pa ON i.id = pa.invoice_id
      LEFT JOIN receipts r ON pa.receipt_id = r.id
      GROUP BY m.id
      HAVING total_value > 0
      ON DUPLICATE KEY UPDATE
        time_value = VALUES(time_value),
        expense_value = VALUES(expense_value),
        total_value = VALUES(total_value),
        invoiced_amount = VALUES(invoiced_amount),
        collected_amount = VALUES(collected_amount)
    `;
        const [result] = await pool.query(query, [
            month, startDate, endDate, startDate, endDate,
            startDate, endDate, startDate, endDate,
            startDate, endDate, startDate, endDate, userId
        ]);
        return result.affectedRows;
    }
}
export const wipRepo = new WIPRepository();
//# sourceMappingURL=repo.js.map