import { getPool } from '../../db/pool.js';
export class WorkflowRepository {
    /**
     * Get all workflows
     */
    async findAll() {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT 
        w.id, w.workflow_name, w.description, w.stages,
        w.is_default, w.is_active, w.created_by,
        CONCAT(e.first_name, ' ', e.surname) as created_by_name,
        w.created_at, w.updated_at
       FROM workflow_config w
       LEFT JOIN users u ON u.id = w.created_by
       LEFT JOIN employees e ON e.id = u.employee_id
       ORDER BY w.is_default DESC, w.workflow_name ASC`);
        return rows.map((row) => ({
            ...row,
            stages: JSON.parse(row.stages),
        }));
    }
    /**
     * Find workflow by ID
     */
    async findById(id) {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT 
        w.id, w.workflow_name, w.description, w.stages,
        w.is_default, w.is_active, w.created_by,
        CONCAT(e.first_name, ' ', e.surname) as created_by_name,
        w.created_at, w.updated_at
       FROM workflow_config w
       LEFT JOIN users u ON u.id = w.created_by
       LEFT JOIN employees e ON e.id = u.employee_id
       WHERE w.id = ?`, [id]);
        if (rows.length === 0)
            return null;
        return {
            ...rows[0],
            stages: JSON.parse(rows[0].stages),
        };
    }
    /**
     * Create workflow
     */
    async create(data, userId) {
        const pool = await getPool();
        // If setting as default, unset other defaults first
        if (data.is_default) {
            await pool.query(`UPDATE workflow_config SET is_default = 0`);
        }
        const [result] = await pool.query(`INSERT INTO workflow_config (workflow_name, description, stages, is_default, created_by)
       VALUES (?, ?, ?, ?, ?)`, [
            data.workflow_name,
            data.description || null,
            JSON.stringify(data.stages),
            data.is_default ? 1 : 0,
            userId,
        ]);
        return result.insertId;
    }
    /**
     * Update workflow
     */
    async update(id, data) {
        const pool = await getPool();
        const fields = [];
        const values = [];
        if (data.workflow_name !== undefined) {
            fields.push('workflow_name = ?');
            values.push(data.workflow_name);
        }
        if (data.description !== undefined) {
            fields.push('description = ?');
            values.push(data.description);
        }
        if (data.stages !== undefined) {
            fields.push('stages = ?');
            values.push(JSON.stringify(data.stages));
        }
        if (data.is_default !== undefined) {
            // If setting as default, unset others first
            if (data.is_default) {
                await pool.query(`UPDATE workflow_config SET is_default = 0`);
            }
            fields.push('is_default = ?');
            values.push(data.is_default ? 1 : 0);
        }
        if (data.is_active !== undefined) {
            fields.push('is_active = ?');
            values.push(data.is_active ? 1 : 0);
        }
        if (fields.length === 0)
            return false;
        values.push(id);
        const [result] = await pool.query(`UPDATE workflow_config SET ${fields.join(', ')} WHERE id = ?`, values);
        return result.affectedRows > 0;
    }
    /**
     * Transition matter to new stage
     */
    async transitionStage(matterId, data, userId) {
        const pool = await getPool();
        // Get matter's workflow to determine stage order and SLA
        const [matter] = await pool.query(`SELECT workflow_id, current_stage FROM cases WHERE id = ?`, [matterId]);
        if (matter.length === 0) {
            throw new Error('Matter not found');
        }
        let stageOrder = 0;
        let slaHours = null;
        if (matter[0].workflow_id) {
            const [workflow] = await pool.query(`SELECT stages FROM workflow_config WHERE id = ?`, [matter[0].workflow_id]);
            if (workflow.length > 0) {
                const stages = JSON.parse(workflow[0].stages);
                const stageConfig = stages.find((s) => s.name === data.stage);
                if (stageConfig) {
                    stageOrder = stageConfig.order;
                    slaHours = stageConfig.sla_hours;
                }
            }
        }
        // Close previous stage
        if (matter[0].current_stage) {
            await pool.query(`UPDATE matter_stages SET ended_at = NOW() 
         WHERE matter_id = ? AND stage = ? AND ended_at IS NULL`, [matterId, matter[0].current_stage]);
        }
        // Calculate SLA due date
        let slaDueAt = null;
        if (slaHours) {
            const dueDate = new Date();
            dueDate.setHours(dueDate.getHours() + slaHours);
            slaDueAt = dueDate.toISOString().slice(0, 19).replace('T', ' ');
        }
        // Create new stage record
        const [result] = await pool.query(`INSERT INTO matter_stages (
        matter_id, stage, stage_order, sla_hours, sla_due_at,
        transition_reason, notes, transitioned_by
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?)`, [
            matterId,
            data.stage,
            stageOrder,
            slaHours,
            slaDueAt,
            data.transition_reason || null,
            data.notes || null,
            userId,
        ]);
        // Update matter's current stage
        await pool.query(`UPDATE cases SET current_stage = ? WHERE id = ?`, [data.stage, matterId]);
        return result.insertId;
    }
    /**
     * Get stage history for a matter
     */
    async getMatterStages(matterId) {
        const pool = await getPool();
        const [rows] = await pool.query(`SELECT 
        ms.id, ms.matter_id, ms.stage, ms.stage_order,
        ms.started_at, ms.ended_at, ms.sla_hours, ms.sla_due_at,
        ms.is_overdue, ms.transition_reason, ms.notes,
        ms.transitioned_by,
        CONCAT(e.first_name, ' ', e.surname) as transitioned_by_name,
        ms.created_at
       FROM matter_stages ms
       LEFT JOIN users u ON u.id = ms.transitioned_by
       LEFT JOIN employees e ON e.id = u.employee_id
       WHERE ms.matter_id = ?
       ORDER BY ms.stage_order ASC, ms.started_at ASC`, [matterId]);
        return rows;
    }
    /**
     * Get all overdue matters
     */
    async getOverdueMatters() {
        const pool = await getPool();
        // First, mark overdue stages
        await pool.query(`UPDATE matter_stages 
       SET is_overdue = 1
       WHERE ended_at IS NULL 
         AND sla_due_at IS NOT NULL 
         AND sla_due_at < NOW()
         AND is_overdue = 0`);
        // Get overdue matters
        const [rows] = await pool.query(`SELECT 
        ms.matter_id,
        c.matter_id as matter_number,
        c.title,
        c.case_type,
        ms.stage,
        ms.sla_due_at,
        TIMESTAMPDIFF(HOUR, ms.sla_due_at, NOW()) as hours_overdue,
        cl.name as client_name,
        CONCAT(e.first_name, ' ', e.surname) as lawyer_name
       FROM matter_stages ms
       JOIN cases c ON c.id = ms.matter_id
       JOIN clients cl ON cl.id = c.client_id
       JOIN lawyers l ON l.id = c.lawyer_id
       JOIN employees e ON e.id = l.employee_id
       WHERE ms.ended_at IS NULL
         AND ms.is_overdue = 1
         AND c.is_active = 1
       ORDER BY ms.sla_due_at ASC`);
        return rows;
    }
}
//# sourceMappingURL=repo.js.map