import { Router } from 'express';
import { WorkflowRepository } from './repo.js';
import { createWorkflowSchema, updateWorkflowSchema, getWorkflowSchema, transitionStageSchema, getMatterStagesSchema, } from './dto.js';
import { validate } from '../../middleware/validate.js';
import { authenticate } from '../../middleware/auth.js';
import { requirePermission } from '../../middleware/rbac.js';
import { logger } from '../../utils/logger.js';
const router = Router();
const repo = new WorkflowRepository();
// All routes require authentication
router.use(authenticate);
/**
 * GET /api/workflows
 * List all workflow templates
 */
router.get('/', requirePermission('workflows:read'), async (req, res, next) => {
    try {
        const workflows = await repo.findAll();
        res.json({
            success: true,
            data: { workflows },
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id }, 'Failed to list workflows');
        next(error);
    }
});
/**
 * GET /api/workflows/:id
 * Get workflow by ID
 */
router.get('/:id', requirePermission('workflows:read'), validate(getWorkflowSchema), async (req, res, next) => {
    try {
        const { id } = req.params;
        const workflow = await repo.findById(parseInt(id));
        if (!workflow) {
            return res.status(404).json({
                success: false,
                error: 'Workflow not found',
            });
        }
        res.json({
            success: true,
            data: workflow,
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id, workflowId: req.params.id }, 'Failed to get workflow');
        next(error);
    }
});
/**
 * POST /api/workflows
 * Create a new workflow template
 */
router.post('/', requirePermission('workflows:manage'), validate(createWorkflowSchema), async (req, res, next) => {
    try {
        const data = req.body;
        const userId = req.user.id;
        const workflowId = await repo.create(data, userId);
        // Log audit
        logger.info({
            userId,
            action: 'CREATE_WORKFLOW',
            workflowId,
            workflowName: data.workflow_name,
        }, 'Workflow created');
        const workflow = await repo.findById(workflowId);
        res.status(201).json({
            success: true,
            message: 'Workflow created successfully',
            data: workflow,
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id }, 'Failed to create workflow');
        next(error);
    }
});
/**
 * PUT /api/workflows/:id
 * Update workflow template
 */
router.put('/:id', requirePermission('workflows:manage'), validate(updateWorkflowSchema), async (req, res, next) => {
    try {
        const { id } = req.params;
        const data = req.body;
        const updated = await repo.update(parseInt(id), data);
        if (!updated) {
            return res.status(404).json({
                success: false,
                error: 'Workflow not found or no changes made',
            });
        }
        // Log audit
        logger.info({
            userId: req.user.id,
            action: 'UPDATE_WORKFLOW',
            workflowId: id,
            changes: data,
        }, 'Workflow updated');
        const workflow = await repo.findById(parseInt(id));
        res.json({
            success: true,
            message: 'Workflow updated successfully',
            data: workflow,
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id, workflowId: req.params.id }, 'Failed to update workflow');
        next(error);
    }
});
/**
 * POST /api/matters/:id/stage
 * Transition matter to new stage
 */
router.post('/matters/:id/stage', requirePermission('workflows:transition'), validate(transitionStageSchema), async (req, res, next) => {
    try {
        const { id } = req.params;
        const data = req.body;
        const userId = req.user.id;
        const stageId = await repo.transitionStage(parseInt(id), data, userId);
        // Log audit
        logger.info({
            userId,
            action: 'TRANSITION_STAGE',
            matterId: id,
            stage: data.stage,
            reason: data.transition_reason,
        }, 'Matter stage transitioned');
        res.status(201).json({
            success: true,
            message: `Matter transitioned to ${data.stage}`,
            data: { stageId },
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id, matterId: req.params.id }, 'Failed to transition stage');
        next(error);
    }
});
/**
 * GET /api/matters/:id/stages
 * Get stage history for a matter
 */
router.get('/matters/:id/stages', requirePermission('matters:read'), validate(getMatterStagesSchema), async (req, res, next) => {
    try {
        const { id } = req.params;
        const stages = await repo.getMatterStages(parseInt(id));
        res.json({
            success: true,
            data: { stages },
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id, matterId: req.params.id }, 'Failed to get matter stages');
        next(error);
    }
});
/**
 * GET /api/workflows/overdue
 * Get all overdue matters
 */
router.get('/stages/overdue', requirePermission('workflows:read'), async (req, res, next) => {
    try {
        const overdueMatters = await repo.getOverdueMatters();
        res.json({
            success: true,
            data: { overdueMatters },
        });
    }
    catch (error) {
        logger.error({ error, userId: req.user?.id }, 'Failed to get overdue matters');
        next(error);
    }
});
export default router;
//# sourceMappingURL=routes.js.map