import express from 'express';
import cookieParser from 'cookie-parser';
import cors from 'cors';
import rateLimit from 'express-rate-limit';
import { config } from './config.js';
import { logger, createRequestLogger } from './utils/logger.js';
import { securityHeaders } from './security/headers.js';
import { errorHandler, notFoundHandler } from './middleware/errors.js';
import { createPool } from './db/pool.js';
const app = express();
// Security middleware
app.use(securityHeaders);
app.use(cors({
    origin: config.CORS_ORIGIN,
    credentials: true,
}));
// Rate limiting (more permissive in development)
const limiter = rateLimit({
    windowMs: config.RATE_LIMIT_WINDOW_MS,
    max: config.NODE_ENV === 'development' ? 1000 : config.RATE_LIMIT_MAX_REQUESTS,
    message: {
        success: false,
        error: 'Too many requests, please try again later.',
    },
    standardHeaders: true,
    legacyHeaders: false,
    skip: (req) => config.NODE_ENV === 'development' && req.path === '/health',
});
app.use('/api', limiter);
// Body parsing
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));
app.use(cookieParser());
// Request logging
app.use(createRequestLogger());
// Health check endpoint
app.get('/health', async (req, res) => {
    try {
        const pool = await createPool();
        await pool.query('SELECT 1');
        res.json({
            success: true,
            data: {
                status: 'healthy',
                timestamp: new Date().toISOString(),
                environment: config.NODE_ENV,
            },
        });
    }
    catch (error) {
        res.status(503).json({
            success: false,
            error: 'Service unavailable',
        });
    }
});
// API routes
import authRoutes from './modules/auth/routes.js';
import departmentRoutes from './modules/departments/routes.js';
import roleRoutes from './modules/roles/routes.js';
import employeeRoutes from './modules/employees/routes.js';
import userRoutes from './modules/users/routes.js';
import dashboardRoutes from './modules/dashboard/routes.js';
import matterRoutes from './modules/matters/routes.js';
import conflictRoutes from './modules/conflicts/routes.js';
import workflowRoutes from './modules/workflows/routes.js';
import courtRoutes from './modules/courts/routes.js';
import evidenceRoutes from './modules/evidence/routes.js';
import timeRoutes from './modules/time/routes.js';
import expenseRoutes from './modules/expenses/routes.js';
import feeRoutes from './modules/fees/routes.js';
import invoiceRoutes from './modules/invoices/routes.js';
import receiptRoutes from './modules/receipts/routes.js';
import trustRoutes from './modules/trust/routes.js';
import wipRoutes from './modules/wip/routes.js';
app.use('/api/auth', authRoutes);
app.use('/api/departments', departmentRoutes);
app.use('/api/roles', roleRoutes);
app.use('/api/employees', employeeRoutes);
app.use('/api/users', userRoutes);
app.use('/api/dashboard', dashboardRoutes);
app.use('/api/matters', matterRoutes);
app.use('/api/conflicts', conflictRoutes);
app.use('/api/workflows', workflowRoutes);
app.use('/api/courts', courtRoutes);
app.use('/api/evidence', evidenceRoutes);
app.use('/api/time', timeRoutes);
app.use('/api/expenses', expenseRoutes);
app.use('/api/fees', feeRoutes);
app.use('/api/invoices', invoiceRoutes);
app.use('/api/receipts', receiptRoutes);
app.use('/api/trust', trustRoutes);
app.use('/api/wip', wipRoutes);
// Error handlers
app.use(notFoundHandler);
app.use(errorHandler);
// Start server
async function startServer() {
    try {
        // Initialize database connection
        await createPool();
        logger.info('Database connection pool initialized');
        const server = app.listen(config.PORT, config.HOST, () => {
            logger.info({
                port: config.PORT,
                host: config.HOST,
                env: config.NODE_ENV,
            }, `Server running on http://${config.HOST}:${config.PORT}`);
            logger.info('Press CTRL+C to stop');
        });
        // Graceful shutdown
        const shutdown = async (signal) => {
            logger.info(`${signal} received. Starting graceful shutdown...`);
            server.close(() => {
                logger.info('HTTP server closed');
            });
            // Close database connections
            const { closePool } = await import('./db/pool.js');
            await closePool();
            logger.info('Graceful shutdown complete');
            process.exit(0);
        };
        process.on('SIGTERM', () => shutdown('SIGTERM'));
        process.on('SIGINT', () => shutdown('SIGINT'));
    }
    catch (error) {
        logger.error({ err: error }, 'Failed to start server');
        process.exit(1);
    }
}
startServer();
export default app;
//# sourceMappingURL=server.js.map